//
//  OPItem.h
//  OnePasswordiOS
//
//  Created by Roustem Karimov on 12-02-19.
//  Copyright (c) 2012 AgileBits. All rights reserved.
//

#import <Foundation/Foundation.h>

#import <AgileLibrary/AgileLibrary.h>
#import <OnePasswordCore/OnePasswordCore.h>

#import <OnePasswordDataModel/OPDatabaseObjectWithOverview.h>
#import <OnePasswordDataModel/OPProfile.h>

#import <OnePasswordDataModel/OPItemField.h>
#import <OnePasswordDataModel/OPFieldSection.h>
#import <OnePasswordDataModel/OPItemReferenceField.h>
#import <OnePasswordDataModel/OPItemURLField.h>
#import <OnePasswordDataModel/OPItemGenderField.h>
#import <OnePasswordDataModel/OPItemAddressField.h>
#import <OnePasswordDataModel/OPItemCreditCardTypeField.h>
#import <OnePasswordDataModel/OPItemPhoneField.h>
#import <OnePasswordDataModel/OPItemConcealedField.h>
#import <OnePasswordDataModel/OPItemDateField.h>
#import <OnePasswordDataModel/OPItemCountryField.h>
#import <OnePasswordDataModel/OPItemMonthField.h>
#import <OnePasswordDataModel/OPItemMonthYearField.h>
#import <OnePasswordDataModel/OPItemEmailField.h>

#import <OnePasswordDataModel/OPPreviouslyUsedPassword.h>

// Item Scope
FOUNDATION_EXPORT NSString *const OPItemScopeAlways;
FOUNDATION_EXPORT NSString *const OPItemScopeRegular;
FOUNDATION_EXPORT NSString *const OPItemScopeBasicAuth;
FOUNDATION_EXPORT NSString *const OPItemScopeNever;


#define MAX_CUSTOM_ICON_SIZE 16384

#if TARGET_OS_IPHONE
#define DETAIL_ICON_SIZE 60.0
#define CUSTOM_DETAIL_ICON_SIZE 64.0
#define CUSTOM_LIST_ICON_SIZE 40.0
#else
#define DETAIL_ICON_SIZE 57.0
#define CUSTOM_DETAIL_ICON_SIZE DETAIL_ICON_SIZE
#define CUSTOM_LIST_ICON_SIZE 32.0
#endif

#define LIST_ICON_SIZE 32.0
#define MINI_ICON_SIZE 16.0

extern NSString *const OPAppleWatchTag;

@class OPItemURL;
@class OPItemAttachment;
@class OPTombstone;
@class OPLogin;
@class B5ItemProperties;

@interface OPItem : OPDatabaseObjectWithOverview <OPSyncable, OPDisplayOrderable, NSSecureCoding>

@property (nonatomic) NSString * title;
@property (nonatomic) NSString * notesPlain;
@property (nonatomic) NSString * URLString;
@property (nonatomic, readonly) NSString * nakedDomain;
@property (nonatomic, readonly) short passwordStrength;

@property (nonatomic) NSString * additionalInformation;

@property (nonatomic) OPAESKey * key;
@property (nonatomic) NSSet * tags;
@property (nonatomic, readonly) id<OPGenericCategory> category;

//
// Backup Keys
// The API is intentionally made to be difficult to remove keys
@property (nonatomic, readonly) NSSet *backupKeysBase64Encoded; // NSStrings
@property (readonly) NSSet *backupAESKeys; // OPAESKeys
- (void)addBackupKeysBase64Encoded:(NSSet *)backupKeysBase64Encoded; // won't mark item dirty if it already contains these backup keys.
- (void)test_only_clearAndDisableBackupKeys; // Disable Backup Keys so that we can run tests without having random keys living in the details dict
- (void)disableBackupKeys; // Will avoid populating backupKeys further and removes its own key from backupKeys if present
- (void)enableBackupKeys;
- (BOOL)isBackupKeysDisabled;

- (void)test_only_disableSpotlightSearch; // Disable Spotlight Search so that we can run tests without having random keys living in the details dict

//
// Transient properties used by advanced search on Mac
//
@property (nonatomic, assign) int32_t numberOfAttachments;
@property (nonatomic, assign) int32_t numberOfTimesUsed;
@property (nonatomic, assign) OPTimestamp lastUsedTimestamp;
@property (nonatomic, assign) int8_t vulnerabilityAction;

//
// database properties
//
@property (nonatomic, strong) NSString * uuid; // Do not access directly. Use -updateUuid:ifProfileTypeMatches: if necessary.
@property (nonatomic) NSData * encryptedKey;
@property (nonatomic) OPTimestamp createdAt;
@property (nonatomic) OPTimestamp updatedAt;
@property (nonatomic) OPID folderId;
@property (nonatomic, copy) NSString * folderUUID;
@property (nonatomic) OPID categoryId;
@property (nonatomic, strong) NSString * categoryUUID;
@property (nonatomic, strong) NSString * sectionName;
@property (nonatomic, strong) NSString * scope;
@property (nonatomic, assign, getter = isShownInBrowser) BOOL shownInBrowser;
@property (nonatomic, strong) NSData * detailsData;
@property (nonatomic) NSUInteger faveIndex;
@property (nonatomic, readonly, getter = isFavorite) BOOL favorite;

@property (nonatomic, readonly, strong) B5ItemProperties *b5ItemProperties; // Will return nil if profileType is not B5


+ (OPItem *)itemWithAttributes:(NSDictionary *)attributes forProfile:(id<OPGenericProfile>)profile;

+ (NSArray *)sortAlphabetically:(NSArray *)items;
+ (NSArray *)sortAlphabetically:(NSArray *)items favoritesOnTop:(BOOL)favoritesOnTop;

+ (NSString *)firstLetterOfString:(NSString *)string;

+ (NSCharacterSet *)searchSeparatorSet;
+ (id)jsonFromFieldSections:(NSArray *)fieldSections;
+ (NSArray *)fieldSectionsFromJSON:(id)json;

- (void)regenerateUuid;
- (void)updateUuid:(NSString *)uuid ifProfileTypeMatches:(OPProfileType)profileType; // Don't pass in receiver.profileType as the second argument. that defeats the purpose of this protection.

- (BOOL)isModifiedCopyOf:(id)anotherObject;
- (NSDictionary *)details;
- (NSDictionary *)detailsCopy;
- (void)setDetailsObject:(id)object forKey:(NSString *)key;
- (void)copyDetails:(NSDictionary *)dict;
- (void)clearFolder;
- (NSString *)json;

- (OPTombstone *)makeTombstone;
- (OPItem *)makeCopyWithoutAttachments;
- (OPItem *)makeDuplicateWithoutAttachments;

- (NSString *)lowercaseTitle;
- (NSString *)additionalInformation;

- (BOOL)hasRegularScope;
- (BOOL)hasNeverScope;
- (NSString *)localizedScopeDescription;
+ (NSString *)localizedScopeDescriptionForValue:(NSString *)value;

- (NSString *)descriptionWithSizeInformation;

- (BOOL)hasFieldWithIdentifierPrefix:(NSString *)prefix;

- (BOOL)isTaggedForAppleWatch;
- (void)toggleTagForAppleWatch;

#if TARGET_OS_IPHONE
- (NSDictionary *)appleWatchDictionaryRepresentation;
#endif // TARGET_OS_IPHONE

// Decryption - Will try to decrypt data with the main key first, then the backup keys.
- (NSData *)decryptUsingAvailableKeys:(NSData *)encryptedData;

- (BOOL)decryptExternalAttachmentUsingAvailableKeys:(OPItemAttachment *)attachment destinationURL:(NSURL *)destinationURL;

- (BOOL)decryptExternalAttachmentUsingAvailableKeys:(OPItemAttachment *)attachment destinationData:(NSMutableData *)data;

// Decrypts external and non-internal attachments to a fileURL
- (BOOL)decryptAttachmentOfAnyKindUsingAvailableKeys:(OPItemAttachment *)attachment destinationURL:(NSURL *)destinationURL;

// Icons
- (OPImage *)detailIcon;
- (OPImage *)listIcon;
- (OPImage *)miniIcon;

- (OPImage *)frameCustomDetailIcon:(OPImage *)icon scale:(CGFloat)scale;
- (OPImage *)frameCustomListIcon:(OPImage *)icon scale:(CGFloat)scale;
- (OPImage *)frameCustomMiniIcon:(OPImage *)icon scale:(CGFloat)scale;

// Tags
- (BOOL)hasTags;
- (NSUInteger)numberOfTags;
- (void)addTag:(NSString *)tag;
- (void)removeTag:(NSString *)tag;
- (NSString *)tagsPlain;

// Login Fields
- (BOOL)hasLoginFields;

// URLs
- (BOOL)hasURLs;
- (NSArray *)URLs;
- (NSArray *)deletedURLs;
- (NSUInteger)numberOfURLs;
- (OPItemURL *)URLAtIndex:(NSUInteger)index;
- (OPItemURL *)URLWithLabel:(NSString *)label;
- (OPItemURL *)URLWithURLString:(NSString *)URLString;
- (void)addURLString:(NSString *)URLString;
- (void)addURLString:(NSString *)URLString withLabel:(NSString *)label;
- (void)addURL:(OPItemURL *)URL;
- (BOOL)hasURLWithString:(NSString *)URLString;
- (void)removeURLAtIndex:(NSUInteger)index;
- (void)moveURL:(OPItemURL *)URL toIndex:(NSUInteger)index;
- (NSString *)defaultLabelForNewURL;
+ (NSString *)goAndFillURLString:(NSString *)string uuid:(NSString *)uuid profileUUID:(NSString *)profileUUID;
- (NSString *)goAndFillURLString;

// Fields
- (BOOL)hasSectionForConflicts;
- (OPFieldSection *)sectionForConflictsCreatedIfNeeded;
- (OPFieldSection *)sectionForReferencesCreatedIfNeeded;
- (NSArray *)fieldSections;
- (NSInteger)numberOfFieldSections;
- (OPFieldSection *)findSectionWithName:(NSString *)name;
- (OPFieldSection *)findSectionWithTitle:(NSString *)title;
- (OPFieldSection *)fieldSectionWithName:(NSString *)name title:(NSString *)title;
- (OPFieldSection *)fieldSectionAtIndex:(NSUInteger)index;
- (void)addFieldSection:(OPFieldSection *)group;
- (void)removeFieldSection:(OPFieldSection *)section;
- (void)removeFieldSectionAtIndex:(NSUInteger)index;
- (void)moveField:(OPItemField *)field fromSection:(OPFieldSection *)fromSection toSection:(OPFieldSection *)toSection toIndex:(NSUInteger)toIndex;
- (id)findFieldWithIdentifier:(NSString *)name;
- (id)findFieldWithValue:(id)rawValue;
- (id)findFieldEqualToField:(OPItemField *)field;
- (OPItemField *)findNonEmptyConcealedField;
- (void)addFieldSections:(NSArray *)fieldSections;
- (void)_setFieldSections:(NSArray *)fieldSections;

// Attachments
- (NSArray *)attachments;
- (NSArray *)sortedAttachments;
- (NSArray *)attachmentsIncludeTombstones:(BOOL)includeTombstones;
- (NSArray *)sortedAttachmentsIncludeTombstones:(BOOL)includeTombstones;
- (NSArray *)attachmentsAwaitingBurial;
- (OPItemAttachment *)attachmentWithUUID:(NSString *)attachmentUUID;
- (void)addAttachment:(OPItemAttachment *)attachment;
- (void)buryAttachment:(OPItemAttachment *)attachment;

- (BOOL)containsPasswordValue:(NSString *)password;

// ReferenceFields
- (void)stripReferenceFields;
- (BOOL)hasReferenceFields;

// Password History
- (BOOL)hasPreviouslyUsedPasswords;
- (NSArray *)previouslyUsedPasswords;
- (NSUInteger)numberOfPreviouslyUsedPasswords;
- (void)addPreviouslyUsedPassword:(OPPreviouslyUsedPassword *)password;
- (void)removePreviouslyUsedPasswordAtIndex:(NSInteger)index;
- (OPPreviouslyUsedPassword *)previouslyUsedPasswordAtIndex:(NSUInteger)index;
- (OPPreviouslyUsedPassword *)previouslyUsedPasswordWithValue:(NSString *)value;
- (OPTimestamp)lastPasswordChange;
- (void)noteOriginalPassword;

- (BOOL)titleMatchesLowercaseStrings:(NSSet *)strings;
- (BOOL)matchesHelperSearchStrings:(NSSet *)strings prefixMatch:(BOOL)prefixMatch;
- (BOOL)overviewMatchesLowercaseStrings:(NSSet *)strings;
- (BOOL)contentsMatchesLowercaseStrings:(NSSet *)strings;
- (BOOL)isSameRevisionAsItem:(OPItem *)item;

// Tools
- (OPLogin *)convertToLogin;  // @protected

// To Override
- (BOOL)hasUsernameOrPassword;
- (BOOL)hasUsername;
- (NSString *)username;
- (void)setUsername:(NSString *)username;

- (BOOL)hasPassword;
- (NSString *)password;
- (void)setPassword:(NSString *)password;

- (NSString *)_additionalInformation;
- (NSString *)brainAction;
- (NSDictionary *)brainDictionary;
- (BOOL)isCreatedByBrowserExtension;
- (NSString *)clipboardValue;

// Search Support
- (NSString *)folderUUIDForSearch;
- (NSArray *)URLStringsForSearch;
- (NSArray *)tagsForSearch;

- (NSArray *)allValuesForSearch;
- (NSArray *)allPasswordsForSearch;
- (NSArray *)allFieldTitlesForSearch;

- (NSDate *)createdAtAsDate;
- (NSDate *)updatedAtAsDate;
- (NSDate *)lastUsedAtAsDate;

- (NSInteger)lastUsedDaysAgo;
- (NSInteger)createdAtDaysAgo;
- (NSInteger)updatedAtDaysAgo;
- (NSInteger)passwordAgeInDays;

@end

